program INTEGRATE;
{--------------------------------------------------------------------}
{  Alg7'12.pas   Pascal program for implementing Algorithm 7.1-2     }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 7.1 (Composite Trapezoidal Rule).                       }
{  Section   7.2, Composite Trapezoidal and Simpson's Rule, Page 365 }
{                                                                    }
{  Algorithm 7.2 (Composite Simpson Rule).                           }
{  Section   7.2, Composite Trapezoidal and Simpson's Rule, Page 365 }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    FunMax = 9;

  type
    LETTER = string[8];
    LETTERS = string[200];
    States = (Changes, Done, Working);
    DoSome = (Go, Stop);
    BIGVEC = array[0..400] of REAL;

  var
    FunType, Inum, M, Meth, Mold, Sub: INTEGER;
    A, B, I, J, Rnum, S2, T2: REAL;
    VF, VX: BIGVEC;
    Ans, Resp: CHAR;
    State: States;
    DoMo: DoSome;
    Mess: LETTERS;

  function F (var X: REAL): REAL;
  begin
    case FunType of
      0:
        begin
          if X <> 0 then
            F := SIN(X) / X
          else
            F := 1;
        end;
      1:
        begin
          if X <> 0 then
            F := 1 / X
          else
            begin
              if A = 0 then
                F := 1E37;
              if B = 0 then
                F := -1E37;
              if A * B < 0 then
                F := 0;
            end;
        end;
      2: 
        F := SIN(X);
      3: 
        F := 4 / (1 + X * X);
      4:
        F := X / (1 + X * X);
      5: 
        begin
          if 2 * X <> 7 then
            F := 1 / (7 - 2 * X)
          else
            begin
              if A = 3.5 then
                F := -1E37;
              if B = 3.5 then
                F := 1E37;
              if (2 * A - 7) * (2 * B - 7) < 0 then
                F := 0;
            end;
        end;
      6: 
        begin
          if X = 3.5 then
            F := -1E37
          else
            F := LN(ABS(X));
        end;
      7: 
        F := X * EXP(-X);
      8: 
        F := 64 * (X - 1) * EXP(-X * LN(4));
      9: 
        F := EXP(-X * X / 2) / SQRT(2 * PI);
    end;
  end;

  procedure PRINTFUNCTION (FunType: INTEGER);
  begin
    case FunType of
      0: 
        WRITE('SIN(X)/X');
      1: 
        WRITE('1/X');
      2:
        WRITE('SIN(X)');
      3: 
        WRITE('4/(1+X^2)');
      4: 
        WRITE('X/(1+X^2)');
      5: 
        WRITE('1/(7-2*X)');
      6: 
        WRITE('LN|X|');
      7: 
        WRITE('X*EXP(-X)');
      8: 
        WRITE('64*(X-1)*4^-X');
      9: 
        WRITE('EXP(-X^2/2)/SQRT(2*Pi)');
    end;
  end;

  function Trule ({FUNCTION F(VAR X:REAL): REAL;}
                  A, B: REAL; M: INTEGER): REAL;
    var
      K: INTEGER;
      H, Sum, X: REAL;
  begin
    H := (B - A) / M;
    Sum := 0;
    for K := 1 to M - 1 do
      begin
        X := A + H * K;
        Sum := Sum + F(X);
      end;
    Sum := H * (F(A) + F(B) + 2 * Sum) / 2;
    Trule := Sum;
  end;

  function Srule ({FUNCTION F(VAR X:REAL): REAL;}
                  A, B: REAL; M: INTEGER): REAL;
    var
      K: INTEGER;
      H, Sum, SumOdd, SumEven, X: REAL;
  begin
    H := (B - A) / (2 * M);
    SumEven := 0;
    for K := 1 to M - 1 do
      begin
        X := A + H * 2 * K;
        SumEven := SumEven + F(X);
      end;
    SumOdd := 0;
    for K := 1 to M do
      begin
        X := A + H * (2 * K - 1);
        SumOdd := SumOdd + F(X);
      end;
    Sum := H * (F(A) + F(B) + 2 * SumEven + 4 * SumOdd) / 3;
    Srule := Sum;
  end;

  function Qrule ({FUNCTION F(VAR X:REAL): REAL;}
                  A, B: REAL; M: INTEGER): REAL;
    var
      K: INTEGER;
      H, Sum, Sum0, Sum1, X, X1, X2: REAL;
  begin
    H := (B - A) / (3 * M);
    Sum0 := 0;
    for K := 1 to M - 1 do
      begin
        X := A + H * 3 * K;
        Sum0 := Sum0 + F(X);
      end;
    Sum1 := 0;
    for K := 0 to M - 1 do
      begin
        X1 := A + H * (3 * K + 1);
        X2 := A + H * (3 * K + 2);
        Sum1 := Sum1 + F(X1) + F(X2);
      end;
    Sum := 3 * H * (F(A) + F(B) + 2 * Sum0 + 3 * Sum1) / 8;
    Qrule := Sum;
  end;

  function Brule ({FUNCTION F(VAR X:REAL): REAL;}
                  A, B: REAL; M: INTEGER): REAL;
    var
      K: INTEGER;
      B2, H, S1, Sum, T0, T1, X: REAL;
   {S2,T2: REAL;}
  begin
    H := (B - A) / M;
    Sum := 0;
    for K := 1 to M - 1 do
      begin
        X := A + H * K;
        Sum := Sum + F(X);
      end;
    T0 := H * (F(A) + F(B) + 2 * Sum) / 2;
    H := H / 2;
    Sum := 0;
    for K := 1 to M do
      begin
        X := A + H * (2 * K - 1);
        Sum := Sum + F(X);
      end;
    T1 := H * Sum + T0 / 2;
    M := 2 * M;
    H := H / 2;
    Sum := 0;
    for K := 1 to M do
      begin
        X := A + H * (2 * K - 1);
        Sum := Sum + F(X);
      end;
    T2 := H * Sum + T1 / 2;
    S1 := (4 * T1 - T0) / 3;
    S2 := (4 * T2 - T1) / 3;
    B2 := (16 * S2 - S1) / 15;
    Brule := B2;
  end;

  function ExtraTrule ({FUNCTION F(VAR X:REAL): REAL;}
                  A, B: REAL; M: INTEGER): REAL;
    var
      K: INTEGER;
      H, Sum, X: REAL;
  begin
    H := (B - A) / M;
    Sum := 0;
    for K := 1 to M - 1 do
      begin
        X := A + H * K;
        VX[K] := X;      {extra output}
        Sum := Sum + F(X);
        VF[K] := H * F(X); {extra output}
      end;
    Sum := H * (F(A) + F(B) + 2 * Sum) / 2;
    ExtraTrule := Sum;
    VF[0] := H * F(A) / 2; {extra output}
    VF[M] := H * F(B) / 2; {extra output}
  end;

  function ExtraSrule ({FUNCTION F(VAR X:REAL): REAL;}
                  A, B: REAL; M: INTEGER): REAL;
    var
      K: INTEGER;
      H, Sum, SumOdd, SumEven, X: REAL;
  begin
    H := (B - A) / (2 * M);
    SumEven := 0;
    for K := 1 to M - 1 do
      begin
        X := A + H * 2 * K;
        VX[2 * K] := X;          {extra output}
        SumEven := SumEven + F(X);
        VF[2 * K] := 2 * H * F(X) / 3; {extra output}
      end;
    SumOdd := 0;
    for K := 1 to M do
      begin
        X := A + H * (2 * K - 1);
        VX[2 * K - 1] := X;          {extra output}
        SumOdd := SumOdd + F(X);
        VF[2 * K - 1] := 4 * H * F(X) / 3; {extra output}
      end;
    Sum := H * (F(A) + F(B) + 2 * SumEven + 4 * SumOdd) / 3;
    ExtraSrule := Sum;
    VF[0] := H * F(A) / 3; {extra output}
    VF[2 * M] := H * F(B) / 3; {extra output}
  end;

  function ExtraQrule ({FUNCTION F(VAR X:REAL): REAL;}
                  A, B: REAL; M: INTEGER): REAL;
    var
      K: INTEGER;
      H, Sum, Sum0, Sum1, X, X1, X2: REAL;
  begin
    H := (B - A) / (3 * M);
    Sum0 := 0;
    for K := 1 to M - 1 do
      begin
        X := A + H * 3 * K;
        VX[3 * K] := X;             {extra output}
        Sum0 := Sum0 + F(X);
        VF[3 * K] := 6 * H * F(X) / 8;    {extra output}
      end;
    Sum1 := 0;
    for K := 0 to M - 1 do
      begin
        X1 := A + H * (3 * K + 1);
        VX[3 * K + 1] := X1;          {extra output}
        X2 := A + H * (3 * K + 2);
        VX[3 * K + 2] := X2;          {extra output}
        Sum1 := Sum1 + F(X1) + F(X2);
        VF[3 * K + 1] := 9 * H * F(X1) / 8; {extra output}
        VF[3 * K + 2] := 9 * H * F(X2) / 8; {extra output}
      end;
    Sum := 3 * H * (F(A) + F(B) + 2 * Sum0 + 3 * Sum1) / 8;
    ExtraQrule := Sum;
    VF[0] := 3 * H * F(A) / 8;        {extra output}
    VF[3 * M] := 3 * H * F(B) / 8;        {extra output}
  end;

  function ExtraBoole ({FUNCTION F(VAR X:REAL): REAL;}
                  A, B: REAL; M: INTEGER): REAL;
    var
      K: INTEGER;
      H, Sum, Sum0, Sum1, Sum2, X, X1, X2, X3: REAL;
  begin
    H := (B - A) / (4 * M);
    Sum0 := 0;
    for K := 1 to M - 1 do
      begin
        X := A + H * (4 * K);
        VX[4 * K] := X;              {extra output}
        Sum0 := Sum0 + F(X);
        VF[4 * K] := 28 * H * F(X) / 45;   {extra output}
      end;
    Sum2 := 0;
    for K := 0 to M - 1 do
      begin
        X := A + H * (4 * K + 2);
        VX[4 * K + 2] := X;            {extra output}
        Sum2 := Sum2 + F(X);
        VF[4 * K + 2] := 24 * H * F(X) / 45; {extra output}
      end;
    Sum1 := 0;
    for K := 0 to M - 1 do
      begin
        X1 := A + H * (4 * K + 1);
        VX[4 * K + 1] := X1;            {extra output}
        X2 := A + H * (4 * K + 3);
        VX[4 * K + 3] := X2;            {extra output}
        Sum1 := Sum1 + F(X1) + F(X2);
        VF[4 * K + 1] := 64 * H * F(X1) / 45; {extra output}
        VF[4 * K + 3] := 64 * H * F(X2) / 45; {extra output}
      end;
    Sum := 2 * H * (7 * F(A) + 7 * F(B) + 14 * Sum0 + 32 * Sum1 + 12 * Sum2) / 45;
    ExtraBoole := Sum;
    VF[0] := 14 * H * F(A) / 45;        {extra output}
    VF[4 * M] := 14 * H * F(B) / 45;        {extra output}
  end;

  procedure MESSAGE (var Meth: INTEGER);
    var
      K: INTEGER;
      Resp: CHAR;
  begin
    CLRSCR;
    WRITELN('                    NUMERICAL INTEGRATION');
    WRITELN;
    WRITELN('         Numerical integration is performed to compute ');
    WRITELN;
    WRITELN('     an approximate value of the definite integral:');
    WRITELN;
    WRITELN('                           B');
    WRITELN('                           /');
    WRITELN('                           | f(x) dx');
    WRITELN('                           /');
    WRITELN('                           A');
    WRITELN;
    WRITELN('                  < 1 > Trapezoidal  Rule');
    WRITELN;
    WRITELN('                  < 2 > Simpson`s Rule');
    WRITELN;
    WRITELN('                  < 3 > Simpson`s 3/8 Rule');
    WRITELN;
    WRITELN('                  < 4 > Boole`s  Rule');
    WRITELN;
    WRITELN('                  < 5 > All of the above.');
    WRITELN;
    Mess := '                        SELECT < 1 - 5 > ?  ';
    Meth := 1;
    WRITE(Mess);
    READLN(Meth);
    if Meth < 1 then
      Meth := 1;
    if Meth > 5 then
      Meth := 5;
  end;

  procedure INPUT (var FunType: INTEGER);
    var
      K: INTEGER;
  begin
    CLRSCR;
    WRITELN('                                                 B');
    WRITELN('                                                 /');
    case Meth of
      1:
        WRITELN('     The Trapezoidal rule is used to approximate | F(X) DX');
      2: 
        WRITELN('     Simpson`s  rule  is  used  to  approximate  | F(X) DX');
      3: 
        WRITELN('     Simpson`s 3/8 rule is used to approximate   | F(X) DX');
      4: 
        WRITELN('     Boole`s  rule  is  used to  approximate     | F(X) DX');
      5: 
        WRITELN('     All the rules will be used to approximate   | F(X) DX');
    end;
    WRITELN('                                                 /');
    WRITELN('                                                 A');
    WRITELN;
    WRITELN('     Choose your function:');
    WRITELN;
    for K := 0 to 9 do
      begin
        WRITE('     <', K : 2, ' >   F(X) = ');
        PRINTFUNCTION(K);
        WRITELN;
      end;
    Mess := '             SELECT < 0 - 9 > ?  ';
    FunType := 0;
    WRITE(Mess);
    READLN(FunType);
    if FunType < 0 then
      FunType := 0;
    if FunType > FunMax then
      FunType := FunMax;
  end;

  procedure PROBLEM (FunType: INTEGER);
  begin
    CLRSCR;
    WRITELN('     You chose to approximate the definite integral:');
    WRITELN;
    WRITELN('                B');
    WRITELN('                /');
    WRITE('                | ');
    PRINTFUNCTION(FunType);
    WRITELN(' DX');
    WRITELN('                /');
    WRITELN('                A');
  end;

  procedure EPOINTS (var A, B: REAL; var M: INTEGER; var State: STATES);
    type
      STATUS = (Change, Enter, Done);
      LETTER = string[1];
    var
      Valu: REAL;
      Resp: CHAR;
      Stat: STATUS;
  begin
    Stat := Enter;
    if State = Changes then
      Stat := Change;
    while (Stat = Enter) or (Stat = Change) do
      begin
        PROBLEM(FunType);
        if (Stat = Enter) then
          begin
            Mess := '               ENTER the left  endpoint    A = ';
            A := 1;
            WRITE(Mess);
            READLN(A);
            Mess := '               ENTER the right endpoint    B = ';
            B := 2;
            WRITE(Mess);
            READLN(B);
            WRITE('     ');
            case Meth of
              1: 
                WRITELN('The number of subintervals used is    M');
              2: 
                WRITELN('The number of subintervals used is  2*M');
              3: 
                WRITELN('The number of subintervals used is  3*M');
              4: 
                WRITELN('The number of subintervals used is  4*M');
              5: 
                WRITELN('The number of subintervals used is 12*M');
            end;
            Mess := '                       ENTER the number    M = ';
            M := 1;
            WRITE(Mess);
            READLN(M);
            if M < 1 then
              M := 1;
            if M > 2050 then
              M := 2050;
          end
        else
          begin
            WRITELN('                  The left  endpoint is    A = ', A : 15 : 7);
            WRITELN;
            WRITELN('                  The right endpoint is    B = ', B : 15 : 7);
            WRITELN;
            WRITE('     ');
            case Meth of
              1: 
                WRITELN('The number of subintervals used is    M =  ', M);
              2: 
                WRITELN('The number of subintervals used is  2*M =  ', 2 * M);
              3: 
                WRITELN('The number of subintervals used is  3*M =  ', 3 * M);
              4: 
                WRITELN('The number of subintervals used is  4*M =  ', 4 * M);
              5: 
                WRITELN('The number of subintervals used is 12*M =  ', 12 * M);
            end;
          end;
        WRITE('        Do you want to make a change ?  <Y/N>  ');
        READ(Resp);
        WRITELN;
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Change;
            PROBLEM(FunType);
            WRITELN('     The current left  endpoint is A =', A : 15 : 7);
            Mess := '     ENTER  the NEW left  endpoint A = ';
            WRITE(Mess);
            READLN(A);
            WRITELN('     The current right endpoint is B =', B : 15 : 7);
            Mess := '     ENTER  the NEW right endpoint B = ';
            WRITE(Mess);
            READLN(B);
            WRITELN('     The  current value of  M  is  M = ', M);
            Mess := '     ENTER  the   NEW   value  of  M = ';
            WRITE(Mess);
            READLN(M);
            if (M < 1) then
              M := 1;
            if (M > 2050) then
              M := 2050;
          end
        else
          Stat := Done;
      end;
  end;

  procedure RESULTS (A, B, I: REAL; M: INTEGER);
    var
      Sum: REAL;
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN(B : 13 : 5);
    WRITELN('       /');
    WRITE('       |  ');
    PRINTFUNCTION(FunType);
    WRITELN(' DX ~', I : 15 : 8);
    WRITELN('       /');
    WRITELN(A : 13 : 5);
    WRITELN;
    case Meth of
      1: 
        WRITELN('     The Trapezoidal rule was used to approximate');
      2: 
        WRITELN('     Simpson`s rule was used to approximate');
      3: 
        WRITELN('     Simpson`s 3/8 rule was used to approximate');
      4: 
        WRITELN('     Boole`e rule was used to approximate');
      5: 
        WRITE('     All the methods were was used to approximate');
    end;
    WRITELN;
    WRITELN('the value of the definite integral:');
    WRITELN;
    WRITE('     F(X) = ');
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN;
    WRITELN('Taken over the interval   [', A : 15 : 8, '  ,', B : 15 : 8, '  ].');
    WRITELN;
    WRITELN('     The value M = ', M : 2, ', was chosen.');
    WRITELN;
    case Meth of
      1: 
        begin
          if M = 1 then
            WRITE('Hence, there was  1  subinterval used in')
          else
            WRITE('Hence, there were ', M : 2, ' subintervals used in');
        end;
      2: 
        WRITE('Hence, there were ', 2 * M : 2, '  subintervals used in');
      3: 
        WRITE('Hence, there were ', 3 * M : 2, '  subintervals used in');
      4: 
        WRITE('Hence, there were ', 4 * M : 2, '  subintervals used in');
      5: 
        WRITE('Hence, there were ', 12 * M : 2, '  subintervals used in');
    end;
    if (M > 1) or (Meth > 1) then
      begin
        WRITELN(' the');
        if Meth <> 5 then
          WRITELN;
        WRITE('composite ');
      end
    else
      begin
        if Meth = 1 then
          WRITELN(' the')
        else
          WRITELN;
        WRITELN;
      end;
    case Meth of
      1: 
        WRITELN('Trapezoidal rule to obtain the approximation  ', I : 15 : 8);
      2: 
        WRITELN('Simpson`s rule to obtain the approximation  ', I : 15 : 8);
      3: 
        WRITELN('Simpson`s 3/8 rule to obtain the approximation  ', I : 15 : 8);
      4: 
        WRITELN('Boole`s rule to obtain the approximation  ', I : 15 : 8);
      5: 
        begin
          WRITELN('integration formulas to get the:');
          WRITELN;
          WRITELN('Trapezoidal  rule  approximation  ', T2 : 15 : 8);
          WRITELN('Simpson`s  rule    approximation  ', S2 : 15 : 8);
          WRITELN('Simpson`s 3/8 rule approximation  ', J : 15 : 8);
          WRITELN('Boole`s   rule     approximation  ', I : 15 : 8);
        end;
    end;
  end;

  procedure PRINTAPPROX;
    var
      K: INTEGER;
      I: REAL;
      Ans: CHAR;
  begin
    CLRSCR;
    if Meth = 1 then
      begin
        I := ExtraTrule(A, B, M);
        WRITELN;
        WRITELN(' a = ', A, '           h f(a)/2 = ', VF[0] : 17 : 8);
        WRITELN;
        for K := 1 to M - 1 do
          begin
            WRITELN('x  = ', VX[K], '           h f(x  ) = ', VF[K] : 17 : 8);
            if K < 10 then
              WRITELN(' ', K : 1, '                                    ', K : 1)
            else
              WRITELN(' ', K : 1, '                                   ', K : 1);
            if K mod 11 = 9 then
              begin
                WRITE('                  Press the  <ENTER>  key.  ');
                READLN(Ans);
                WRITELN;
              end;
          end;
        WRITELN(' b = ', B, '           h f(b)/2 = ', VF[M] : 17 : 8);
        WRITELN;
        WRITELN('The  trapezoidal  rule  approximation  is   ', I : 17 : 8);
        WRITELN;
        WRITE('                  Press the <ENTER> key.  ');
        READLN(Ans);
        WRITELN;
      end;
    if Meth = 2 then
      begin
        I := ExtraSrule(A, B, M);
        WRITELN;
        WRITELN(' a = ', A, '         h f( a )/3 = ', VF[0] : 17 : 8);
        WRITELN;
        for K := 1 to M - 1 do
          begin
            WRITELN('x  = ', VX[2 * K - 1], '        4h f(x  )/3 = ', VF[2 * K - 1] : 17 : 8);
            if (2 * K - 1) < 10 then
              WRITELN(' ', 2 * K - 1 : 1, '                                  ', 2 * K - 1 : 1)
            else
              WRITELN(' ', 2 * K - 1 : 2, '                                 ', 2 * K - 1 : 1);
            WRITELN('x  = ', VX[2 * K], '        2h f(x  )/3 = ', VF[2 * K] : 17 : 8);
            if (2 * K) < 10 then
              WRITELN(' ', 2 * K : 1, '                                  ', 2 * K : 1)
            else
              WRITELN(' ', 2 * K : 2, '                                 ', 2 * K : 1);
            if K mod 5 = 0 then
              begin
                WRITE('                  Press the  <ENTER>  key.  ');
                READLN(Ans);
                WRITELN;
              end;
          end;
        WRITELN('x  = ', VX[2 * M - 1], '        4h f(x  )/3 = ', VF[2 * M - 1] : 17 : 8);
        if (2 * M - 1) < 10 then
          WRITELN(' ', 2 * M - 1 : 1, '                                  ', 2 * M - 1 : 1)
        else
          WRITELN(' ', 2 * M - 1 : 2, '                                 ', 2 * M - 1 : 1);
        WRITELN(' b = ', B, '         h f( b )/3 = ', VF[2 * M] : 17 : 8);
        WRITELN;
        WRITELN('The   Simpson   rule   approximation   is   ', I : 17 : 8);
        WRITELN;
        WRITE('                  Press the  <ENTER>  key.  ');
        READLN(Ans);
        WRITELN;
      end;
    if Meth = 3 then
      begin
        I := ExtraQrule(A, B, M);
        WRITELN;
        WRITELN(' a = ', A, '       3 h f( a )/8 = ', VF[0] : 17 : 8);
        WRITELN;
        for K := 1 to M - 1 do
          begin
            WRITELN('x  = ', VX[3 * K - 2], '       9 h f(x  )/8 = ', VF[3 * K - 2] : 17 : 8);
            if (3 * K - 2) < 10 then
              WRITELN(' ', 3 * K - 2 : 1, '                                  ', 3 * K - 2 : 1)
            else
              WRITELN(' ', 3 * K - 2 : 1, '                                 ', 3 * K - 2 : 1);
            WRITELN('x  = ', VX[3 * K - 1], '       9 h f(x  )/8 = ', VF[3 * K - 1] : 17 : 8);
            if (3 * K - 1) < 10 then
              WRITELN(' ', 3 * K - 1 : 1, '                                  ', 3 * K - 1 : 1)
            else
              WRITELN(' ', 3 * K - 1 : 1, '                                 ', 3 * K - 1 : 1);
            WRITELN('x  = ', VX[3 * K], '       3 h f(x  )/4 = ', VF[3 * K] : 17 : 8);
            if (3 * K) < 10 then
              WRITELN(' ', 3 * K : 1, '                                  ', 3 * K : 1)
            else
              WRITELN(' ', 3 * K : 1, '                                 ', 3 * K : 1);
            if K mod 3 = 0 then
              begin
                WRITE('                  Press the  <ENTER>  key.  ');
                READLN(Ans);
                WRITELN;
              end;
          end;
        WRITELN('x  = ', VX[3 * M - 2], '       9 h f(x  )/8 = ', VF[3 * M - 2] : 17 : 8);
        if (3 * M - 2) < 10 then
          WRITELN(' ', 3 * M - 2 : 1, '                                  ', 3 * M - 2 : 1)
        else
          WRITELN(' ', 3 * M - 2 : 1, '                                 ', 3 * M - 2 : 1);
        WRITELN('x  = ', VX[3 * M - 1], '       9 h f(x  )/8 = ', VF[3 * M - 1] : 17 : 8);
        if (3 * M - 1) < 10 then
          WRITELN(' ', 3 * M - 1 : 1, '                                  ', 3 * M - 1 : 1)
        else
          WRITELN(' ', 3 * M - 1 : 1, '                                 ', 3 * M - 1 : 1);
        WRITELN(' b = ', B, '       3 h f( b )/8 = ', VF[3 * M] : 17 : 8);
        WRITELN;
        WRITELN('Simpson   3/8   rule   approximation   is   ', I : 17 : 8);
        WRITELN;
        WRITE('                  Press the  <ENTER>  key.  ');
        READLN(Ans);
        WRITELN;
      end;
    if Meth = 4 then
      begin
        I := ExtraBoole(A, B, M);
        WRITELN;
        WRITELN(' a = ', A, '      14 h f( a )/45 = ', VF[0] : 17 : 8);
        WRITELN;
        for K := 1 to M - 1 do
          begin
            WRITELN('x  = ', VX[4 * K - 3], '      64 h f(x  )/45 = ', VF[4 * K - 3] : 17 : 8);
            if (4 * K - 3) < 10 then
              WRITELN(' ', 4 * K - 3 : 1, '                                  ', 4 * K - 3 : 1)
            else
              WRITELN(' ', 4 * K - 3 : 1, '                                 ', 4 * K - 3 : 1);
            WRITELN('x  = ', VX[4 * K - 2], '      24 h f(x  )/45 = ', VF[4 * K - 2] : 17 : 8);
            if (4 * K - 2) < 10 then
              WRITELN(' ', 4 * K - 2 : 1, '                                  ', 4 * K - 2 : 1)
            else
              WRITELN(' ', 4 * K - 2 : 1, '                                 ', 4 * K - 2 : 1);
            WRITELN('x  = ', VX[4 * K - 1], '      64 h f(x  )/45 = ', VF[4 * K - 1] : 17 : 8);
            if (4 * K - 1) < 10 then
              WRITELN(' ', 4 * K - 1 : 1, '                                  ', 4 * K - 1 : 1)
            else
              WRITELN(' ', 4 * K - 1 : 1, '                                 ', 4 * K - 1 : 1);
            WRITELN('x  = ', VX[4 * K], '      28 h f(x  )/45 = ', VF[4 * K] : 17 : 8);
            if (4 * K) < 10 then
              WRITELN(' ', 4 * K : 1, '                                  ', 4 * K : 1)
            else
              WRITELN(' ', 4 * K : 1, '                                 ', 4 * K : 1);
            if K mod 2 = 0 then
              begin
                WRITE('                  Press the  <ENTER>  key.  ');
                READLN(Ans);
                WRITELN;
              end;
          end;
        WRITELN('x  = ', VX[4 * M - 3], '      64 h f(x  )/45 = ', VF[4 * M - 3] : 17 : 8);
        if (4 * M - 3) < 10 then
          WRITELN(' ', 4 * M - 3 : 1, '                                  ', 4 * M - 3 : 1)
        else
          WRITELN(' ', 4 * M - 3 : 1, '                                 ', 4 * M - 3 : 1);
        WRITELN('x  = ', VX[4 * M - 2], '      24 h f(x  )/45 = ', VF[4 * M - 2] : 17 : 8);
        if (4 * M - 2) < 10 then
          WRITELN(' ', 4 * M - 2 : 1, '                                  ', 4 * M - 2 : 1)
        else
          WRITELN(' ', 4 * M - 2 : 1, '                                 ', 4 * M - 2 : 1);
        WRITELN('x  = ', VX[4 * M - 1], '      64 h f(x  )/45 = ', VF[4 * M - 1] : 17 : 8);
        if (4 * M - 1) < 10 then
          WRITELN(' ', 4 * M - 1 : 1, '                                  ', 4 * M - 1 : 1)
        else
          WRITELN(' ', 4 * M - 1 : 1, '                                 ', 4 * M - 1 : 1);
        WRITELN(' b = ', B, '      14 h f( b )/45 = ', VF[4 * M] : 17 : 8);
        WRITELN;
        WRITELN('The   Boole`s   rule   approximation   is    ', I : 17 : 8);
        WRITELN;
        WRITE('                  Press the  <ENTER>  key.  ');
        READLN(Ans);
        WRITELN;
      end;
  end;

begin                                            {Begin Main Program}
  Meth := 2;
  while Meth <> 0 do
    begin
      MESSAGE(Meth);
      DoMo := Go;
      while DoMo = Go do
        begin
          INPUT(FunType);
          State := Working;
          while (State = Working) or (State = Changes) do
            begin
              EPOINTS(A, B, M, State);
              case Meth of
                1:
                  I := Trule(A, B, M);
                2:
                  I := Srule(A, B, M);
                3:
                  I := Qrule(A, B, M);
                4:
                  I := Brule(A, B, M);
                5:
                  begin
                    I := Brule(A, B, 3 * M);
                    J := Qrule(A, B, 4 * M);
                  end;
              end;
              RESULTS(A, B, I, M);
              WRITELN;
              WRITE('Want  to see  the summation terms ?  <Y/N>  ');
              READLN(Resp);
              Resp := 'y';
              if ((Resp = 'Y') or (Resp = 'y')) and (Meth < 5) then
                PRINTAPPROX;
              if ((Resp = 'Y') or (Resp = 'y')) and (Meth = 5) then
                begin
                  Mold := M;
                  Meth := 1;
                  M := 12 * Mold;
                  PRINTAPPROX;
                  Meth := 2;
                  M := 6 * Mold;
                  PRINTAPPROX;
                  Meth := 3;
                  M := 4 * Mold;
                  PRINTAPPROX;
                  Meth := 4;
                  M := 3 * Mold;
                  PRINTAPPROX;
                  Meth := 5;
                  M := Mold;
                end;
              WRITELN;
              WRITE('Want  to  try   another  interval ?  <Y/N>  ');
              READLN(Ans);
              WRITELN;
              if (Ans <> 'Y') and (Ans <> 'y') then
                State := Done;
            end;
          WRITELN;
          WRITE('Want to use a  different function ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans <> 'Y') and (Ans <> 'y') then
            DoMo := Stop;
        end;
      WRITELN;
      WRITE('Do you want to try another method ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        Meth := 0;
    end;                                          {End of Main Program}
end.

